<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package Business_Shuffle
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function business_shuffle_body_classes($classes) {
    // Adds a class of hfeed to non-singular pages.
    if (!is_singular()) {
        $classes[] = 'hfeed';
    }

    return $classes;
}

add_filter('body_class', 'business_shuffle_body_classes');

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function business_shuffle_pingback_header() {
    if (is_singular() && pings_open()) {
        printf('<link rel="pingback" href="%s">', esc_url(get_bloginfo('pingback_url')));
    }
}

add_action('wp_head', 'business_shuffle_pingback_header');

if (!function_exists('business_shuffle_add_search_block')) {

    function business_shuffle_add_search_block($items, $args) {
        $enable_search = get_theme_mod('business_shuffle_mh_show_search', false);

        if ($enable_search) {
            if ($args->theme_location == 'business-shuffle-menu') {
                $items .= '<li class="menu-item menu-item-search"><a href="javascript:void(0)"><i class="fas fa-search"></i></a></li>';
            }
        }
        return $items;
    }

}

add_filter('wp_nav_menu_items', 'business_shuffle_add_search_block', 10, 2);

if (!function_exists('business_shuffle_excerpt')) {

    function business_shuffle_excerpt($content, $letter_count) {
        $content = strip_shortcodes($content);
        $content = strip_tags($content);
        $content = mb_substr($content, 0, $letter_count);

        if (strlen($content) == $letter_count) {
            $content .= "...";
        }
        return $content;
    }

}

/* Convert hexdec color string to rgb(a) string */

function business_shuffle_hex2rgba($color, $opacity = false) {

    $default = 'rgb(0,0,0)';

    //Return default if no color provided
    if (empty($color))
        return $default;

    //Sanitize $color if "#" is provided 
    if ($color[0] == '#') {
        $color = substr($color, 1);
    }

    //Check if color has 6 or 3 characters and get values
    if (strlen($color) == 6) {
        $hex = array($color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5]);
    } elseif (strlen($color) == 3) {
        $hex = array($color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2]);
    } else {
        return $default;
    }

    //Convert hexadec to rgb
    $rgb = array_map('hexdec', $hex);

    //Check if opacity is set(rgba or rgb)
    if ($opacity) {
        if (abs($opacity) > 1)
            $opacity = 1.0;
        $output = 'rgba(' . implode(",", $rgb) . ',' . $opacity . ')';
    } else {
        $output = 'rgb(' . implode(",", $rgb) . ')';
    }

    //Return rgb(a) color string
    return $output;
}

function business_shuffle_color_brightness($hex, $percent) {
    // Work out if hash given
    $hash = '';
    if (stristr($hex, '#')) {
        $hex = str_replace('#', '', $hex);
        $hash = '#';
    }
    /// HEX TO RGB
    $rgb = array(hexdec(substr($hex, 0, 2)), hexdec(substr($hex, 2, 2)), hexdec(substr($hex, 4, 2)));
    //// CALCULATE 
    for ($i = 0; $i < 3; $i++) {
        // See if brighter or darker
        if ($percent > 0) {
            // Lighter
            $rgb[$i] = round($rgb[$i] * $percent) + round(255 * (1 - $percent));
        } else {
            // Darker
            $positivePercent = $percent - ($percent * 2);
            $rgb[$i] = round($rgb[$i] * $positivePercent) + round(0 * (1 - $positivePercent));
        }
        // In case rounding up causes us to go to 256
        if ($rgb[$i] > 255) {
            $rgb[$i] = 255;
        }
    }
    //// RBG to Hex
    $hex = '';
    for ($i = 0; $i < 3; $i++) {
        // Convert the decimal digit to hex
        $hexDigit = dechex($rgb[$i]);
        // Add a leading zero if necessary
        if (strlen($hexDigit) == 1) {
            $hexDigit = "0" . $hexDigit;
        }
        // Append to the hex string
        $hex .= $hexDigit;
    }
    return $hash . $hex;
}

function business_shuffle_css_strip_whitespace($css) {
    $replace = array(
        "#/\*.*?\*/#s" => "", // Strip C style comments.
        "#\s\s+#" => " ", // Strip excess whitespace.
    );
    $search = array_keys($replace);
    $css = preg_replace($search, $replace, $css);

    $replace = array(
        ": " => ":",
        "; " => ";",
        " {" => "{",
        " }" => "}",
        ", " => ",",
        "{ " => "{",
        ";}" => "}", // Strip optional semicolons.
        ",\n" => ",", // Don't wrap multiple selectors.
        "\n}" => "}", // Don't wrap closing braces.
        "} " => "}\n", // Put each rule on it's own line.
    );
    $search = array_keys($replace);
    $css = str_replace($search, $replace, $css);

    return trim($css);
}

if (!function_exists('business_shuffle_home_section')) {

    function business_shuffle_home_section() {
        $defaults = apply_filters('business_shuffle_home_sections', array(
            'business_shuffle_hightlight_section',
            'business_shuffle_about_section',
            'business_shuffle_progressbar_section',
            'business_shuffle_featured_section',
            'business_shuffle_cta_section',
            'business_shuffle_counter_section',
            'business_shuffle_blog_section',
            'business_shuffle_logo_section'
                )
        );
        $sections = get_theme_mod('business_shuffle_frontpage_sections', $defaults);
        return $sections;
    }

}

function business_shuffle_comment($comment, $args, $depth) {
    extract($args, EXTR_SKIP);
    $tag = ( 'div' === $args['style'] ) ? 'div' : 'li';
    $show_avatars = get_option('show_avatars');
    ?>
    <<?php echo $tag; ?> id="comment-<?php comment_ID(); ?>" <?php comment_class(empty($args['has_children']) ? 'parent' : '', $comment); ?>>
    <article id="div-comment-<?php comment_ID(); ?>" class="comment-body">
        <?php if (0 != $args['avatar_size'] && $show_avatars) { ?>
            <div class="bs-comment-avatar">
                <?php echo get_avatar($comment, $args['avatar_size']); ?>
            </div>
        <?php } ?>

        <div class="bs-comment-content">
            <div class="comment-header">
                <div class="comment-author vcard">
                    <?php
                    echo sprintf('<b class="fn">%s</b>', get_comment_author_link($comment));
                    echo " - ";
                    ?>
                    <a href="<?php echo esc_url(get_comment_link($comment, $args)); ?>">
                        <time>
                            <?php
                            echo get_comment_date('', $comment);
                            ?>
                        </time>
                    </a>
                </div>

                <?php
                comment_reply_link(array_merge($args, array(
                    'add_below' => 'div-comment',
                    'depth' => $depth,
                    'max_depth' => $args['max_depth'],
                    'before' => '<div class="reply">',
                    'after' => '</div>'
                )));
                ?>

                <!-- .comment-author -->

                <?php if ('0' == $comment->comment_approved) : ?>
                    <p class="comment-awaiting-moderation">
                        <?php esc_html_e('Your comment is awaiting moderation.', 'business-shuffle'); ?>
                    </p>
                <?php endif; ?>

            </div>
            <!-- .comment-meta -->

            <div class="comment-content">
                <?php comment_text(); ?>
            </div>
            <!-- .comment-content -->
            <?php edit_comment_link(esc_html__('Edit', 'business-shuffle'), '<div class="edit-link">', '</div>'); ?>
        </div>
        <!-- .comment-metadata -->
    </article>
    <!-- .comment-body -->
    <?php
}
