<?php

/**
 * Business Shuffle functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Business_Shuffle
 */
if (!function_exists('business_shuffle_setup')) :

    /**
     * Sets up theme defaults and registers support for various WordPress features.
     *
     * Note that this function is hooked into the after_setup_theme hook, which
     * runs before the init hook. The init hook is too late for some features, such
     * as indicating support for post thumbnails.
     */
    function business_shuffle_setup() {
        /*
         * Make theme available for translation.
         * Translations can be filed in the /languages/ directory.
         * If you're building a theme based on Business Shuffle, use a find and replace
         * to change 'business-shuffle' to the name of your theme in all the template files.
         */
        load_theme_textdomain('business-shuffle', get_template_directory() . '/languages');

        // Add default posts and comments RSS feed links to head.
        add_theme_support('automatic-feed-links');

        /*
         * Let WordPress manage the document title.
         * By adding theme support, we declare that this theme does not use a
         * hard-coded <title> tag in the document head, and expect WordPress to
         * provide it for us.
         */
        add_theme_support('title-tag');

        /*
         * Enable support for Post Thumbnails on posts and pages.
         *
         * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
         */
        add_theme_support('post-thumbnails');
        add_image_size('business-shuffle-400x400', 400, 400, true);
        add_image_size('business-shuffle-150x100', 150, 100, true);

        // This theme uses wp_nav_menu() in one location.
        register_nav_menus(array(
            'business-shuffle-menu' => esc_html__('Primary', 'business-shuffle'),
        ));

        /*
         * Switch default core markup for search form, comment form, and comments
         * to output valid HTML5.
         */
        add_theme_support('html5', array(
            'search-form',
            'comment-form',
            'comment-list',
            'gallery',
            'caption',
        ));

        // Set up the WordPress core custom background feature.
        add_theme_support('custom-background', apply_filters('business_shuffle_custom_background_args', array(
            'default-color' => 'ffffff',
            'default-image' => '',
        )));

        // Add theme support for selective refresh for widgets.
        add_theme_support('customize-selective-refresh-widgets');

        /**
         * Add support for core custom logo.
         *
         * @link https://codex.wordpress.org/Theme_Logo
         */
        add_theme_support('custom-logo', array(
            'height' => 50,
            'width' => 200,
            'flex-width' => true,
            'flex-height' => true,
        ));

        add_post_type_support('page', 'excerpt');
    }

endif;
add_action('after_setup_theme', 'business_shuffle_setup');

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function business_shuffle_content_width() {
    // This variable is intended to be overruled from themes.
    // Open WPCS issue: {@link https://github.com/WordPress-Coding-Standards/WordPress-Coding-Standards/issues/1043}.
    // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
    $GLOBALS['content_width'] = apply_filters('business_shuffle_content_width', 640);
}

add_action('after_setup_theme', 'business_shuffle_content_width', 0);

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function business_shuffle_widgets_init() {
    register_sidebar(array(
        'name' => esc_html__('Right Sidebar', 'business-shuffle'),
        'id' => 'business-shuffle-right-sidebar',
        'description' => esc_html__('Add widgets here.', 'business-shuffle'),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget' => '</section>',
        'before_title' => '<h4 class="widget-title">',
        'after_title' => '</h4>',
    ));

    register_sidebar(array(
        'name' => esc_html__('Footer One', 'business-shuffle'),
        'id' => 'business-shuffle-footer1',
        'description' => esc_html__('Add widgets here.', 'business-shuffle'),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget' => '</section>',
        'before_title' => '<h4 class="widget-title">',
        'after_title' => '</h4>',
    ));

    register_sidebar(array(
        'name' => esc_html__('Footer Two', 'business-shuffle'),
        'id' => 'business-shuffle-footer2',
        'description' => esc_html__('Add widgets here.', 'business-shuffle'),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget' => '</section>',
        'before_title' => '<h4 class="widget-title">',
        'after_title' => '</h4>',
    ));

    register_sidebar(array(
        'name' => esc_html__('Footer Three', 'business-shuffle'),
        'id' => 'business-shuffle-footer3',
        'description' => esc_html__('Add widgets here.', 'business-shuffle'),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget' => '</section>',
        'before_title' => '<h4 class="widget-title">',
        'after_title' => '</h4>',
    ));

    register_sidebar(array(
        'name' => esc_html__('Footer Four', 'business-shuffle'),
        'id' => 'business-shuffle-footer4',
        'description' => esc_html__('Add widgets here.', 'business-shuffle'),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget' => '</section>',
        'before_title' => '<h4 class="widget-title">',
        'after_title' => '</h4>',
    ));
}

add_action('widgets_init', 'business_shuffle_widgets_init');

if (!function_exists('business_shuffle_fonts_url')) :

    /**
     * Register Google fonts for Business Shuffle.
     *
     * @since Business Shuffle 1.0
     *
     * @return string Google fonts URL for the theme.
     */
    function business_shuffle_fonts_url() {
        $fonts_url = '';
        $fonts = array();
        $subsets = 'latin,latin-ext';

        /*
         * Translators: If there are characters in your language that are not supported
         * by Open Sans, translate this to 'off'. Do not translate into your own language.
         */
        if ('off' !== _x('on', 'Poppins font: on or off', 'business-shuffle')) {
            $fonts[] = 'Poppins:300,300i,400,400i,600,600i,700,700i';
        }

        /*
         * Translators: If there are characters in your language that are not supported
         * by Inconsolata, translate this to 'off'. Do not translate into your own language.
         */
        if ('off' !== _x('on', 'Teko font: on or off', 'business-shuffle')) {
            $fonts[] = 'Teko:400,500,600,700';
        }

        /*
         * Translators: To add an additional character subset specific to your language,
         * translate this to 'greek', 'cyrillic', 'devanagari' or 'vietnamese'. Do not translate into your own language.
         */
        $subset = _x('no-subset', 'Add new subset (greek, cyrillic, devanagari, vietnamese)', 'business-shuffle');

        if ('cyrillic' == $subset) {
            $subsets .= ',cyrillic,cyrillic-ext';
        } elseif ('greek' == $subset) {
            $subsets .= ',greek,greek-ext';
        } elseif ('devanagari' == $subset) {
            $subsets .= ',devanagari';
        } elseif ('vietnamese' == $subset) {
            $subsets .= ',vietnamese';
        }

        if ($fonts) {
            $fonts_url = add_query_arg(array(
                'family' => urlencode(implode('|', $fonts)),
                'subset' => urlencode($subsets),
                    ), '//fonts.googleapis.com/css');
        }

        return esc_url_raw($fonts_url);
    }

endif;

/**
 * Enqueue scripts and styles.
 */
function business_shuffle_scripts() {
    wp_enqueue_style('business-shuffle-fonts', business_shuffle_fonts_url(), array(), null);
    wp_enqueue_style('fontawesome', get_template_directory_uri() . '/css/fontawesome.css', array(), '5.2.0');
    wp_enqueue_style('owl-carousel', get_template_directory_uri() . '/css/owl.carousel.css', array(), '2.3.4');
    wp_enqueue_style('business-shuffle-style', get_stylesheet_uri());
    wp_add_inline_style('business-shuffle-style', business_shuffle_dynamic_style());

    wp_enqueue_script('waypoint', get_template_directory_uri() . '/js/waypoint.js', array(), '1.0.0', true);
    wp_enqueue_script('owl-carousel', get_template_directory_uri() . '/js/owl.carousel.js', array('jquery'), '2.3.4', true);
    wp_enqueue_script('superfish', get_template_directory_uri() . '/js/superfish.js', array('jquery'), '1.0', true);
    wp_enqueue_script('business-shuffle-custom-scripts', get_template_directory_uri() . '/js/custom-scripts.js', array('jquery'), '1.0.0', true);

    if (is_singular() && comments_open() && get_option('thread_comments')) {
        wp_enqueue_script('comment-reply');
    }
}

add_action('wp_enqueue_scripts', 'business_shuffle_scripts');

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer/customizer.php';

/**
 * Dynamic Styles.
 */
require get_template_directory() . '/inc/style.php';

/**
 * Load WooCommerce compatibility file.
 */
if (class_exists('WooCommerce')) {
    require get_template_directory() . '/inc/woocommerce.php';
}

/**
 * Widgets additions.
 */
require get_template_directory() . '/inc/widgets/widget-fields.php';
require get_template_directory() . '/inc/widgets/widget-contact-info.php';
require get_template_directory() . '/inc/widgets/widget-personal-info.php';
require get_template_directory() . '/inc/widgets/widget-latest-post.php';

/**
 * 5.5 wpColorPicker Fix
 *
 * @since 1.0.6
 */
if (is_admin()) {
    add_action('wp_default_scripts', 'wp_default_custom_scripts');

    function wp_default_custom_scripts($scripts) {
        $scripts->add('wp-color-picker', "/wp-admin/js/color-picker.js", array('iris'), false, 1);
        did_action('init') && $scripts->localize(
                        'wp-color-picker', 'wpColorPickerL10n', array(
                    'clear' => __('Clear', 'business-shuffle'),
                    'clearAriaLabel' => __('Clear color', 'business-shuffle'),
                    'defaultString' => __('Default', 'business-shuffle'),
                    'defaultAriaLabel' => __('Select default color', 'business-shuffle'),
                    'pick' => __('Select Color', 'business-shuffle'),
                    'defaultLabel' => __('Color value', 'business-shuffle'),
                        )
        );
    }

}